import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Volume2, Lightbulb, Sofa, Package, Calendar, Award, Truck, Headphones, CreditCard, RotateCcw } from 'lucide-react'
import './BookRental.css'

const BookRental = () => {
  const navigate = useNavigate()
  const [selectedCategory, setSelectedCategory] = useState(null)
  const [startDate, setStartDate] = useState('')
  const [endDate, setEndDate] = useState('')

  const categories = [
    { 
      id: 'sound', 
      name: 'Звуковое оборудование', 
      icon: Volume2, 
      color: '#6366f1',
      items: ['Микрофоны', 'Колонки', 'Микшерный пульт', 'Усилители']
    },
    { 
      id: 'lighting', 
      name: 'Освещение', 
      icon: Lightbulb, 
      color: '#8b5cf6',
      items: ['Прожекторы', 'Светодиодные панели', 'Световые эффекты', 'Сценическое освещение']
    },
    { 
      id: 'furniture', 
      name: 'Мебель', 
      icon: Sofa, 
      color: '#ec4899',
      items: ['Столы', 'Стулья', 'Диваны', 'Стеллажи']
    }
  ]

  const calculateTotal = () => {
    if (!selectedCategory || !startDate || !endDate) return 0
    const start = new Date(startDate)
    const end = new Date(endDate)
    const days = Math.ceil((end - start) / (1000 * 60 * 60 * 24)) + 1
    const basePrice = selectedCategory === 'sound' ? 5000 : selectedCategory === 'lighting' ? 4000 : 3000
    return days * basePrice
  }

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedCategory || !startDate || !endDate) {
      alert('Пожалуйста, заполните все поля')
      return
    }

    const start = new Date(startDate)
    const end = new Date(endDate)
    
    if (end <= start) {
      alert('Дата окончания должна быть позже даты начала')
      return
    }

    const rentals = JSON.parse(localStorage.getItem('equipmentRentals') || '{"active": [], "past": []}')
    const category = categories.find(c => c.id === selectedCategory)
    
    const newRental = {
      id: Date.now(),
      category: category.name,
      items: category.items,
      startDate: startDate,
      endDate: endDate,
      status: 'Активна',
      total: calculateTotal()
    }

    rentals.active.push(newRental)
    localStorage.setItem('equipmentRentals', JSON.stringify(rentals))
    
    alert('Аренда успешно оформлена!')
    navigate('/')
  }

  const selectedCategoryData = categories.find(c => c.id === selectedCategory)

  return (
    <div className="book-rental">
      <div className="container">
        <section className="booking-hero">
          <div className="booking-hero-content">
            <div className="booking-hero-top">
              <div className="booking-hero-badge">Оформление аренды</div>
              <h1 className="booking-hero-title">Арендуйте профессиональное оборудование</h1>
              <p className="booking-hero-subtitle">Выберите категорию оборудования, укажите даты аренды и получите все необходимое для вашего мероприятия</p>
            </div>
            <div className="booking-hero-features">
              <div className="booking-feature">
                <div className="booking-feature-dot"></div>
                <span>Широкий ассортимент</span>
              </div>
              <div className="booking-feature">
                <div className="booking-feature-dot"></div>
                <span>Доставка и установка</span>
              </div>
              <div className="booking-feature">
                <div className="booking-feature-dot"></div>
                <span>Гибкие условия</span>
              </div>
            </div>
          </div>
        </section>

        <form className="booking-form" onSubmit={handleSubmit}>
          <div className="form-section">
            <h2 className="section-title">
              <Package className="section-icon" size={28} />
              Выберите категорию оборудования
            </h2>
            <div className="categories-grid">
              {categories.map(category => {
                const IconComponent = category.icon
                return (
                  <button
                    key={category.id}
                    type="button"
                    className={`category-card ${selectedCategory === category.id ? 'selected' : ''}`}
                    onClick={() => setSelectedCategory(category.id)}
                    style={{ '--category-color': category.color }}
                  >
                    <div className="category-icon">
                      <IconComponent className="category-icon-img" size={48} />
                    </div>
                    <span className="category-name">{category.name}</span>
                  </button>
                )
              })}
            </div>
            {selectedCategoryData && (
              <div className="category-items-preview">
                <div className="preview-label">Включает:</div>
                <div className="preview-items">
                  {selectedCategoryData.items.map((item, idx) => (
                    <span key={idx} className="preview-item">{item}</span>
                  ))}
                </div>
              </div>
            )}
          </div>

          <div className="form-section">
            <h2 className="section-title">
              <Calendar className="section-icon" size={28} />
              Выберите даты аренды
            </h2>
            <div className="dates-container">
              <div className="date-input-group">
                <label className="date-label">Дата начала</label>
                <input
                  type="date"
                  className="date-input"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  min={new Date().toISOString().split('T')[0]}
                />
              </div>
              <div className="date-input-group">
                <label className="date-label">Дата окончания</label>
                <input
                  type="date"
                  className="date-input"
                  value={endDate}
                  onChange={(e) => setEndDate(e.target.value)}
                  min={startDate || new Date().toISOString().split('T')[0]}
                />
              </div>
            </div>
            {startDate && endDate && (
              <div className="rental-summary">
                <div className="summary-row">
                  <span>Период аренды:</span>
                  <span className="summary-value">
                    {Math.ceil((new Date(endDate) - new Date(startDate)) / (1000 * 60 * 60 * 24)) + 1} дней
                  </span>
                </div>
                <div className="summary-row">
                  <span>Стоимость:</span>
                  <span className="summary-value total">{calculateTotal().toLocaleString('ru-RU')} ₽</span>
                </div>
              </div>
            )}
          </div>

          <button type="submit" className="submit-button" disabled={!selectedCategory || !startDate || !endDate}>
            Оформить аренду
          </button>
        </form>

        <section className="benefits-showcase">
          <h2 className="benefits-title">Преимущества</h2>
          <div className="benefits-grid">
            <div className="benefit-card">
              <div className="benefit-icon">
                <Award className="benefit-icon-img" size={36} />
              </div>
              <h3>Качественное оборудование</h3>
              <p>Только проверенные бренды и модели</p>
            </div>
            <div className="benefit-card">
              <div className="benefit-icon">
                <Truck className="benefit-icon-img" size={36} />
              </div>
              <h3>Доставка и установка</h3>
              <p>Быстрая доставка и профессиональная установка</p>
            </div>
            <div className="benefit-card">
              <div className="benefit-icon">
                <Headphones className="benefit-icon-img" size={36} />
              </div>
              <h3>Техническая поддержка</h3>
              <p>Круглосуточная поддержка во время аренды</p>
            </div>
          </div>
        </section>

        <div className="info-section">
          <div className="info-section-header">
            <h2 className="info-title">Полезная информация</h2>
          </div>
          <div className="info-grid">
            <div className="info-card">
              <div className="info-card-icon-wrapper">
                <Truck className="info-card-icon" size={36} />
              </div>
              <h3>Доставка</h3>
              <p>Доставка оборудования осуществляется в день начала аренды. Возможна установка нашими специалистами.</p>
            </div>
            <div className="info-card">
              <div className="info-card-icon-wrapper">
                <CreditCard className="info-card-icon" size={36} />
              </div>
              <h3>Оплата</h3>
              <p>Оплата производится при получении оборудования. Принимаем наличные и банковские карты.</p>
            </div>
            <div className="info-card">
              <div className="info-card-icon-wrapper">
                <RotateCcw className="info-card-icon" size={36} />
              </div>
              <h3>Возврат</h3>
              <p>Оборудование необходимо вернуть в день окончания аренды. При повреждении взимается компенсация.</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}

export default BookRental

